using System.Collections;
using System.Collections.Generic;
using System.Data.SqlClient;
using System.Data.SqlTypes;
using System.Linq;
using System.Threading;
using gov.va.med.vbecs.DAL.HL7.OpenLibrary;
using gov.va.med.vbecs.DAL.HL7.OpenLibrary.Messages;
using gov.va.med.vbecs.DAL.HL7.Parsers;
using gov.va.med.vbecs.DAL.HL7AL;
#if NUNIT
#region Directives
//-----------------------***
using System;
using System.Data;
using NUnit.Framework;
using gov.va.med.vbecs.UnitTests;

using TABLES = gov.va.med.vbecs.Common.VbecsTables;
using SPROCS = gov.va.med.vbecs.Common.VbecsStoredProcs;
using ARTIFICIAL = gov.va.med.vbecs.Common.DatabaseConstants.ArtificialColumnNames;
using COMMON = gov.va.med.vbecs.Common;
using gov.va.med.vbecs.Common;

//-----------------------***
#endregion
namespace gov.va.med.vbecs.DAL.HL7AL.UnitTests
{
    /// <summary>
    /// Summary description for PendingResultsController.
    ///PT_114.02 
    ///PT_114.03
    ///BR_114.01 
    ///BR_114.02
    ///BR_114.03
    ///BR_114.04
    ///  
    /// </summary>
    [TestFixture]
    public class HL7AL_PendingResultsController : BaseTest
    {

        [SetUp]
        protected void SetUp()
        {
            BaseSetUp();
        }

        [TearDown]
        protected void TearDown()
        {
        }

        [Test]
        [ExpectedException(typeof(DataAccessLayerException))]
        public void GetPendingBloodUnitTestExceptionTest()
        {
            string bloodUnitId = string.Empty;
            string divisionCode = null;
            PendingResultsController.GetPendingBloodUnitTest(bloodUnitId, divisionCode, PendingTestStatus.Accepted, 22);
        }

        [Test]
        [ExpectedException(typeof(DataAccessLayerException))]
        public void GetPendingSpecimenTestExceptionTest()
        {
            string specimenUid = string.Empty;
            string divisionCode = null;
            PendingResultsController.GetPendingSpecimenTest(specimenUid, divisionCode, PendingTestStatus.Accepted, 22);
        }


        [Test]
        [ExpectedException(typeof(DataAccessLayerException))]
        public void GetBloodUnitExceptionTest()
        {
            string bloodUnitId = string.Empty;
            string divisionCode = null;
            PendingResultsController.GetBloodUnit(bloodUnitId, divisionCode);
        }

        [Test]
        [ExpectedException(typeof(DataAccessLayerException))]
        public void GetSpecimenExceptionTest()
        {
            string specimenUid = string.Empty;
            string bloodTestPanelName = null;
            string bloodTestName = null;
            string divisionCode = null;
            PendingResultsController.GetSpecimen(specimenUid, bloodTestPanelName, bloodTestName,divisionCode );
        }

        ///// <summary>
        ///// 
        ///// </summary>
        //[Test]
        //public void XMInterpISHandling_Pass()
        //{
        //    List<PendingResultsController.PendingTest> pendingTests = new List<PendingResultsController.PendingTest>();
        //    List<string> errorList = new List<string>();
        //    PendingResultsController.PendingTest pendingTest = new PendingResultsController.PendingTest();
        //    pendingTest.BloodTestTypeName = "AHGInterp";
        //    pendingTest.TestResultId = "Incomp";
        //    pendingTest.OrderableTestName = "XM";
        //    pendingTests.Add(pendingTest);
        //    pendingTest = new PendingResultsController.PendingTest();
        //    pendingTest.BloodTestTypeName = "ISInterp";
        //    pendingTest.TestResultId = "Comp";
        //    pendingTest.OrderableTestName = "XM";
        //    pendingTests.Add(pendingTest);

        //    PendingResultsController.XMInterpISHandling(ref pendingTests, ref errorList);
            
        //    Assert.IsTrue(pendingTests.Exists(x => x.BloodTestTypeName.Contains("XMInterp")) &&
        //                  (pendingTests.Find(x => x.BloodTestTypeName.Contains("XMInterp"))).TestResultId == "I");
        //}

        /// <summary>
        /// 
        /// </summary>
        [Test]
        public void HL7GetXMBySpecimenUIDAndUnitID_Pass()
        {
            BOL.BloodUnit tmpBU = DataCreator.CreateRBCBloodUnit(false);
            DataCreator.CreatePatientAssignment(tmpBU.BloodUnitGuid, Guid.NewGuid(), false);
            DataRow drPS = UnitTestUtil.RunSQLGetDataRow("SELECT TOP 1 * FROM PatientSpecimen ORDER BY LastUpdateDate DESC");
            BOL.Specimen tmpPS = new BOL.Specimen(drPS);

            string specimenUID = tmpPS.SpecimenUid;
            string bloodUnitId = tmpBU.LabelUnitId;
            string divsionCode = this.TestDivisionCode;
            DataTable xmTable = PendingResultsController.HL7GetXMBySpecimenUIDAndUnitID(specimenUID, bloodUnitId, this.TestDivisionCode);

            Assert.IsTrue(xmTable.Rows.Count > 0);

        }

        [Test]
        public void GetUserRecord_Pass()
        {
            List<string> validationErrors = new List<string>();

            DataRow drTest = PendingResultsController.GetUserRecord("DNS   LOHSEG0", ref validationErrors);

            Assert.IsFalse(validationErrors.Count > 0);
            Assert.NotNull(drTest, "Data exists");
            //We MUST have these 2 fields at minimum
            Assert.True(drTest.Table.Columns.Contains(TABLES.VbecsUser.VbecsUserId), TABLES.VbecsUser.VbecsUserId);
            Assert.True(drTest.Table.Columns.Contains(TABLES.VbecsUser.UserEmail), TABLES.VbecsUser.UserEmail);
        }


        [Test]
        public void ProcessPendingTestResults_AntigenProcessing_InValidResults()
        {
            string comment = "Antigen Processing";
            List<PendingResultsController.PendingTest> pendingTests = new List<PendingResultsController.PendingTest>();
            ArrayList sprocArray = new ArrayList();
            ArrayList dtArray = new ArrayList();
            BloodTestPanelName bloodTestPanelName = BloodTestPanelName.UnitABORh;
            Guid resutltSetGuid = Guid.NewGuid();
            PendingResultsController.PendingBloodUnitTest pendingTest =
                new PendingResultsController.PendingBloodUnitTest();
            Guid bloodUnitGuid = Guid.NewGuid();

            pendingTest.BloodUnitGuid = bloodUnitGuid;
            pendingTest.BloodTestTypeName = "AntiB"; 
            pendingTest.TestTechId = "ishdobrak";
            pendingTest.TestDate = HL7DateFormat.ConvertHL7DateTime("20160201160003");
            pendingTest.TestResultText = "4";
            pendingTest.AutoInstrumentName = "ProVueSanDiego";
            pendingTest.DivisionCode = "589";
            pendingTest.ResultSetGuid = resutltSetGuid;
            pendingTest.OrderableTestName = "Antigen Typing";
            pendingTests.Add(pendingTest);

            List<string> validationErrors = PendingResultsController.ProcessPendingTestResults(pendingTests,
                comment, ref sprocArray, ref dtArray, bloodTestPanelName);

            Assert.IsTrue(validationErrors.Count == 1); //fieldmapping failure
        }

        [Test]
        public void ProcessPendingTestResults_AboRhIterpProcessing_ValidResults()
        {
            string comment = "Test Patient ABORh";
            List<PendingResultsController.PendingTest> pendingTests = new List<PendingResultsController.PendingTest>();
            ArrayList sprocArray = new ArrayList();
            ArrayList dtArray = new ArrayList();
            BloodTestPanelName bloodTestPanelName = BloodTestPanelName.UnitABORh;
            Guid resutltSetGuid = Guid.NewGuid();
            PendingResultsController.PendingBloodUnitTest pendingTest =
                new PendingResultsController.PendingBloodUnitTest();
            Guid bloodUnitGuid = Guid.NewGuid();

            pendingTest.BloodUnitGuid = bloodUnitGuid;
            pendingTest.BloodTestTypeName = "ABORhInterp";
            pendingTest.TestTechId = "ishdobrak";
            pendingTest.TestDate = HL7DateFormat.ConvertHL7DateTime("20160201160003");
            pendingTest.TestResultId = pendingTest.TestResultId;
            pendingTest.TestResultText = "A Pos";
            pendingTest.AutoInstrumentName = "ProVueSanDiego";
            pendingTest.DivisionCode = "589";
            pendingTest.ResultSetGuid = resutltSetGuid;
            pendingTest.OrderableTestName = "Patient ABORh";
            pendingTests.Add(pendingTest);

            List<string> validationErrors = PendingResultsController.ProcessPendingTestResults(pendingTests,
                comment, ref sprocArray, ref dtArray, bloodTestPanelName);
            Assert.IsTrue(validationErrors.Count == 0);
        }

        [Test]
        public void GetBloodUnitTest()
        {
            BOL.BloodUnit bloodUnit = DataCreator.CreateRBCBloodUnit(false);
            DataTable bloodDataTable = PendingResultsController.GetBloodUnit(bloodUnit.EyeReadableUnitId, bloodUnit.DivisionCode);

            Assert.NotNull(bloodDataTable, "Data exists");
        }

        [Test]
        public void GetSpecimenTest()
        {
            BOL.OrderedTest specimentOrderedTest = DataCreator.CreateNewPatientDiagnosticOrder(true, Guid.NewGuid(), OrderableTest.ABORh );
            DataTable bloodDataTable = PendingResultsController.GetSpecimen(specimentOrderedTest.Specimen.SpecimenUid, "Patient ABORh", null ,specimentOrderedTest.DivisionCode);

            Assert.NotNull(bloodDataTable, "Data exists");
        }


        [Test]
        public void GetUserRecord_Fail()
        {
            List<string> validationErrors = new List<string>();

            DataRow drTest = DAL.HL7AL.PendingResultsController.GetUserRecord("Potatoes", ref validationErrors);

            Assert.IsTrue(validationErrors.Count > 0);
            Assert.IsNull(drTest, "Data does not exist");
        }

        [Test]
        public void GetTestTypeMappingTest()
        {
            int bloodTestTypoeId = 0;
            int mappedTestTypeId = 0;
            List<string> validationErrors = new List<string>();
            #region success

            bool success =  PendingResultsController.GetTestTypeMapping("Patient ABORh", "AntiA", 1, out bloodTestTypoeId,
                out mappedTestTypeId, ref validationErrors);
            Assert.IsTrue(success);
            Assert.IsTrue(bloodTestTypoeId != 0);
            Assert.IsTrue(mappedTestTypeId == 0);

            
            
            //1	Patient ABORh	AntiD1
            success = PendingResultsController.GetTestTypeMapping("Patient ABORh", "AntiD1", 1, out bloodTestTypoeId,
                out mappedTestTypeId, ref validationErrors);
            Assert.IsTrue(success);
            Assert.IsTrue(bloodTestTypoeId != 0);
            Assert.IsTrue(mappedTestTypeId == 0);

            //1	Patient ABORh	AntiD2
            success = PendingResultsController.GetTestTypeMapping("Patient ABORh", "AntiD2", 1, out bloodTestTypoeId,
                out mappedTestTypeId, ref validationErrors);
            Assert.IsTrue(success);
            Assert.IsTrue(bloodTestTypoeId != 0);
            Assert.IsTrue(mappedTestTypeId == 0);

            //73	Antigen Typing	AntiKp(b)
            //73	Antigen Typing	AntiKp(b)Interp
            success = PendingResultsController.GetTestTypeMapping("Antigen Typing", "AntiKp(b)", 73, out bloodTestTypoeId,
                out mappedTestTypeId, ref validationErrors);
            Assert.IsTrue(success);
            Assert.IsTrue(bloodTestTypoeId != 0);
            Assert.IsTrue(mappedTestTypeId != 0);

            #endregion //success
            #region fail
            //Caps in test type name.
            success = PendingResultsController.GetTestTypeMapping("Patient ABORh", "ANTIA", 1, out bloodTestTypoeId,
                out mappedTestTypeId, ref validationErrors);
            Assert.IsFalse(success);
            Assert.IsFalse(bloodTestTypoeId != 0);
            Assert.IsFalse(mappedTestTypeId != 0);


            //Lowercase in test name.
            success = PendingResultsController.GetTestTypeMapping("patient aborh", "AntiA", 1, out bloodTestTypoeId,
                out mappedTestTypeId, ref validationErrors);
            Assert.IsFalse(success);
            Assert.IsFalse(bloodTestTypoeId != 0);
            Assert.IsFalse(mappedTestTypeId != 0);

            //Invalid orderable test id.
            success = PendingResultsController.GetTestTypeMapping("Patient ABORh", "AntiA", 1000, out bloodTestTypoeId,
                out mappedTestTypeId, ref validationErrors);
            Assert.IsFalse(success);
            Assert.IsFalse(bloodTestTypoeId != 0);
            Assert.IsFalse(mappedTestTypeId != 0);

            #endregion //fail
        }

        [Test]
        public void AntiD1AntiD2CompareTest()
        {
            List<PendingResultsController.PendingTest> pendingTests = new List<PendingResultsController.PendingTest>();
            PendingResultsController.PendingSpecimenTest pendingTest =
            new PendingResultsController.PendingSpecimenTest();
            #region success
            pendingTest.TestResultText = "2";
            pendingTest.OrderableTestName = "Patient ABORh";
            pendingTest.BloodTestTypeName = "AntiD1";
            pendingTests.Add(pendingTest);

            pendingTest = new PendingResultsController.PendingSpecimenTest();
            pendingTest.TestResultText = "2";
            pendingTest.OrderableTestName = "Patient ABORh";
            pendingTest.BloodTestTypeName = "AntiD2";
            pendingTests.Add(pendingTest);

            List<string> validationErrors = new List<string>();
            PendingResultsController.AntiD1AntiD2Compare(ref pendingTests, ref validationErrors);
            Assert.IsTrue(validationErrors.Count == 0);
            Assert.IsTrue(pendingTests.Count == 1);
            Assert.IsTrue(pendingTests[0].BloodTestTypeName == "AntiD");
            Assert.IsTrue(pendingTests[0].TestResultText == "2");

            /*******#2*****/
            pendingTests = new List<PendingResultsController.PendingTest>();
            pendingTest = new PendingResultsController.PendingSpecimenTest();
            pendingTest.TestResultText = "3";
            pendingTest.OrderableTestName = "Patient ABORh";
            pendingTest.BloodTestTypeName = "AntiD1";
            pendingTests.Add(pendingTest);

            pendingTest = new PendingResultsController.PendingSpecimenTest();
            pendingTest.TestResultText = "2";
            pendingTest.OrderableTestName = "Patient ABORh";
            pendingTest.BloodTestTypeName = "AntiD2";
            pendingTests.Add(pendingTest);

            validationErrors = new List<string>();
            PendingResultsController.AntiD1AntiD2Compare(ref pendingTests, ref validationErrors);
            Assert.IsTrue(validationErrors.Count == 0);
            Assert.IsTrue(pendingTests.Count == 1);
            Assert.IsTrue(pendingTests[0].BloodTestTypeName == "AntiD");
            Assert.IsTrue(pendingTests[0].TestResultText == "3");


            #endregion
            #region reject

            pendingTests = new List<PendingResultsController.PendingTest>();
            pendingTest = new PendingResultsController.PendingSpecimenTest();

            pendingTest.TestResultText = "0";
            pendingTest.OrderableTestName = "Patient ABORh";
            pendingTest.BloodTestTypeName = "AntiD1";
            pendingTests.Add(pendingTest);

            pendingTest = new PendingResultsController.PendingSpecimenTest();

            pendingTest.TestResultText = "3";
            pendingTest.OrderableTestName = "Patient ABORh";
            pendingTest.BloodTestTypeName = "AntiD2";
            pendingTests.Add(pendingTest);

            validationErrors = new List<string>();
            PendingResultsController.AntiD1AntiD2Compare(ref pendingTests, ref validationErrors);

            Assert.IsTrue(validationErrors.Count > 0);

            #endregion

        }

        
        [Test]
        public void ResultLookupTest()
        {

            List<string> validationErrors = new List<string>();

            PendingResultsController.PendingTest pendingTest =
            new PendingResultsController.PendingTest();

            #region accept

            pendingTest.TestResultId = "2";
            pendingTest.TestResultText = "2";
            pendingTest.OrderableTestName = "Patient ABORh";
            pendingTest.BloodTestTypeName = "AntiD1";

            PendingResultsController.ResultLookup(pendingTest, ref validationErrors);
            Assert.IsTrue(validationErrors.Count == 0);

            #endregion

            #region reject
            pendingTest =
            new PendingResultsController.PendingTest();
            
            pendingTest.TestResultId = "10";
            pendingTest.TestResultText = "10";
            pendingTest.OrderableTestName = "Patient ABORh";
            pendingTest.BloodTestTypeName = "AntiD1";

            PendingResultsController.ResultLookup(pendingTest, ref validationErrors);
            Assert.IsTrue(validationErrors.Count > 0);

            #endregion

        }

        [Test]
        public void AntigenLookupTest()
        {
            List<string> validationErrors = new List<string>();

            PendingResultsController.PendingTest pendingTest =
            new PendingResultsController.PendingTest();

            #region accept
            //1,2,3,4 --> P
            pendingTest.TestResultText = "2";
            pendingTest.OrderableTestName = "Patient ABORh";
            pendingTest.BloodTestTypeName = "AntiD1";

            pendingTest.TestResultId = PendingResultsController.AntigenLookup(pendingTest, ref validationErrors);
            Assert.IsTrue(validationErrors.Count == 0);
            Assert.IsTrue( pendingTest.TestResultId == "P");

            //0-->N
            pendingTest =
            new PendingResultsController.PendingTest();

            pendingTest.TestResultText = "0";
            pendingTest.OrderableTestName = "Patient ABORh";
            pendingTest.BloodTestTypeName = "AntiD1";

            pendingTest.TestResultId = PendingResultsController.AntigenLookup(pendingTest, ref validationErrors);
            Assert.IsTrue(validationErrors.Count == 0);
            Assert.IsTrue(pendingTest.TestResultId == "N");

            #endregion

            #region reject

            pendingTest =
            new PendingResultsController.PendingTest();

            pendingTest.TestResultText = "10";
            pendingTest.OrderableTestName = "Patient ABORh";
            pendingTest.BloodTestTypeName = "AntiD1";

            pendingTest.TestResultId = PendingResultsController.AntigenLookup(pendingTest, ref validationErrors);
            Assert.IsTrue(validationErrors.Count > 0);

            #endregion
        }

        [Test]
        public void InterpResultLookupTest()
        {
            List<string> validationErrors = new List<string>();

            PendingResultsController.PendingTest pendingTest =
            new PendingResultsController.PendingTest();

            #region accept
            //Positive --> P
            pendingTest.TestResultText = "Positive";
            pendingTest.OrderableTestName = "Patient ABORh";
            pendingTest.BloodTestTypeName = "AntiD1";

            pendingTest.TestResultId = PendingResultsController.InterpResultLookup(pendingTest, ref validationErrors);
            Assert.IsTrue(validationErrors.Count == 0);
            Assert.IsTrue(pendingTest.TestResultId == "P");

            //Pos-->P
            pendingTest =
            new PendingResultsController.PendingTest();

            pendingTest.TestResultText = "Pos";
            pendingTest.OrderableTestName = "Patient ABORh";
            pendingTest.BloodTestTypeName = "AntiD1";

            pendingTest.TestResultId = PendingResultsController.InterpResultLookup(pendingTest, ref validationErrors);
            Assert.IsTrue(validationErrors.Count == 0);
            Assert.IsTrue(pendingTest.TestResultId == "P");

            //P-->P
            pendingTest =
            new PendingResultsController.PendingTest();

            pendingTest.TestResultText = "P";
            pendingTest.OrderableTestName = "Patient ABORh";
            pendingTest.BloodTestTypeName = "AntiD1";

            pendingTest.TestResultId = PendingResultsController.InterpResultLookup(pendingTest, ref validationErrors);
            Assert.IsTrue(validationErrors.Count == 0);
            Assert.IsTrue(pendingTest.TestResultId == "P");
            //Pos-->P
            pendingTest =
            new PendingResultsController.PendingTest();

            pendingTest.TestResultText = "Negative";
            pendingTest.OrderableTestName = "Patient ABORh";
            pendingTest.BloodTestTypeName = "AntiD1";

            pendingTest.TestResultId = PendingResultsController.InterpResultLookup(pendingTest, ref validationErrors);
            Assert.IsTrue(validationErrors.Count == 0);
            Assert.IsTrue(pendingTest.TestResultId == "N");

            //Pos-->P
            pendingTest =
            new PendingResultsController.PendingTest();

            pendingTest.TestResultText = "Neg";
            pendingTest.OrderableTestName = "Patient ABORh";
            pendingTest.BloodTestTypeName = "AntiD1";

            pendingTest.TestResultId = PendingResultsController.InterpResultLookup(pendingTest, ref validationErrors);
            Assert.IsTrue(validationErrors.Count == 0);
            Assert.IsTrue(pendingTest.TestResultId == "N");

            //Pos-->P
            pendingTest =
            new PendingResultsController.PendingTest();

            pendingTest.TestResultText = "N";
            pendingTest.OrderableTestName = "Patient ABORh";
            pendingTest.BloodTestTypeName = "AntiD1";

            pendingTest.TestResultId = PendingResultsController.InterpResultLookup(pendingTest, ref validationErrors);
            Assert.IsTrue(validationErrors.Count == 0);
            Assert.IsTrue(pendingTest.TestResultId == "N");

            #endregion

            #region reject

            pendingTest =
            new PendingResultsController.PendingTest();

            pendingTest.TestResultText = "L";
            pendingTest.OrderableTestName = "Patient ABORh";
            pendingTest.BloodTestTypeName = "AntiD1";

            pendingTest.TestResultId = PendingResultsController.InterpResultLookup(pendingTest, ref validationErrors);
            Assert.IsTrue(validationErrors.Count > 0);

            #endregion
        }

        [Test]
        public void XMInterpResultLookupTest()
        {
            List<string> validationErrors = new List<string>();

            PendingResultsController.PendingTest pendingTest =
            new PendingResultsController.PendingTest();

            #region accept
            //Positive --> P
            pendingTest.TestResultText = "Compatible";
            pendingTest.OrderableTestName = "Patient ABORh";
            pendingTest.BloodTestTypeName = "AntiD1";

            pendingTest.TestResultId = PendingResultsController.XMInterpResultLookup(pendingTest, ref validationErrors);
            Assert.IsTrue(validationErrors.Count == 0);
            Assert.IsTrue(pendingTest.TestResultId == "C");

            //Pos-->P
            pendingTest =
            new PendingResultsController.PendingTest();

            pendingTest.TestResultText = "Comp";
            pendingTest.OrderableTestName = "Patient ABORh";
            pendingTest.BloodTestTypeName = "AntiD1";

            pendingTest.TestResultId = PendingResultsController.XMInterpResultLookup(pendingTest, ref validationErrors);
            Assert.IsTrue(validationErrors.Count == 0);
            Assert.IsTrue(pendingTest.TestResultId == "C");

            //P-->P
            pendingTest =
            new PendingResultsController.PendingTest();

            pendingTest.TestResultText = "C";
            pendingTest.OrderableTestName = "Patient ABORh";
            pendingTest.BloodTestTypeName = "AntiD1";

            pendingTest.TestResultId = PendingResultsController.XMInterpResultLookup(pendingTest, ref validationErrors);
            Assert.IsTrue(validationErrors.Count == 0);
            Assert.IsTrue(pendingTest.TestResultId == "C");
            //Pos-->P
            pendingTest =
            new PendingResultsController.PendingTest();

            pendingTest.TestResultText = "Incompatible";
            pendingTest.OrderableTestName = "Patient ABORh";
            pendingTest.BloodTestTypeName = "AntiD1";

            pendingTest.TestResultId = PendingResultsController.XMInterpResultLookup(pendingTest, ref validationErrors);
            Assert.IsTrue(validationErrors.Count == 0);
            Assert.IsTrue(pendingTest.TestResultId == "I");

            //Pos-->P
            pendingTest =
            new PendingResultsController.PendingTest();

            pendingTest.TestResultText = "Incomp";
            pendingTest.OrderableTestName = "Patient ABORh";
            pendingTest.BloodTestTypeName = "AntiD1";

            pendingTest.TestResultId = PendingResultsController.XMInterpResultLookup(pendingTest, ref validationErrors);
            Assert.IsTrue(validationErrors.Count == 0);
            Assert.IsTrue(pendingTest.TestResultId == "I");

            //Pos-->P
            pendingTest =
            new PendingResultsController.PendingTest();

            pendingTest.TestResultText = "I";
            pendingTest.OrderableTestName = "Patient ABORh";
            pendingTest.BloodTestTypeName = "AntiD1";

            pendingTest.TestResultId = PendingResultsController.XMInterpResultLookup(pendingTest, ref validationErrors);
            Assert.IsTrue(validationErrors.Count == 0);
            Assert.IsTrue(pendingTest.TestResultId == "I");

            #endregion

            #region reject

            pendingTest =
            new PendingResultsController.PendingTest();

            pendingTest.TestResultText = "L";
            pendingTest.OrderableTestName = "Patient ABORh";
            pendingTest.BloodTestTypeName = "AntiD1";

            pendingTest.TestResultId = PendingResultsController.XMInterpResultLookup(pendingTest, ref validationErrors);
            Assert.IsTrue(validationErrors.Count > 0);

            #endregion
        }

        [Test]
        public void XMInterpISHandlingTest()
        {
            //"MSH|^~\&|IM||VBECS|589A7|20160203131726||ORU^R01|1143|P|2.4|12||AL|NE
            //OBR|1|C160350007^W696900000044||XM||||||||||||||ProVueSanDiego
            //OBX|1||IS||0||||||F|||20160203131726||DNS   dobrak
            //OBX|2||AHG||0||||||F|||20160203131726||DNS   dobrak
            //OBX|3||ISInterp||Comp||||||F|||20160203131726||DNS   dobrak
            //OBX|4||AHGInterp||Comp||||||F|||20160203131726||DNS   dobrak
            //OBX|5||Comment||||||||F|||20160203131726||DNS   dobrak
            //"
            List<PendingResultsController.PendingTest> pendingTests = new List<PendingResultsController.PendingTest>();
            PendingResultsController.PendingBloodUnitTest pendingTest =
            new PendingResultsController.PendingBloodUnitTest();
            #region success
            pendingTest.TestResultText = "Comp";
            pendingTest.OrderableTestName = "XM";
            pendingTest.BloodTestTypeName = "ISInterp";
            pendingTests.Add(pendingTest);

            pendingTest = new PendingResultsController.PendingBloodUnitTest();
            pendingTest.TestResultText = "Comp";
            pendingTest.OrderableTestName = "XM";
            pendingTest.BloodTestTypeName = "AHGInterp";
            pendingTests.Add(pendingTest);

            List<string> validationErrors = new List<string>();
            PendingResultsController.XMInterpISHandling(ref pendingTests, ref validationErrors);
            Assert.IsTrue(validationErrors.Count == 0);
            Assert.IsTrue(pendingTests.Count == 4);

            #endregion
        }

        [Test]
        public void AddUpdateOrderedTestTaskStatusTest()
        {
           DataTable updateTaskStatus = PendingResultsController.GetOrderTaskStatusSprocTable();
           PendingResultsController.PendingTest pendingTest = new PendingResultsController.PendingTest();
           List<string> validationErrors = new List<string>();
           BOL.OrderedTest specimentOrderedTest = DataCreator.CreateNewPatientDiagnosticOrder(true, Guid.NewGuid(), OrderableTest.ABORh);

           pendingTest.OrderedTestGuid = specimentOrderedTest.OrderedTestGuid;          
           pendingTest.RowVersion = specimentOrderedTest.RowVersion;
           pendingTest.TestTechId = specimentOrderedTest.LastUpdateUser;

           int rowCount = updateTaskStatus.Rows.Count;


            PendingResultsController.AddUpdateOrderedTestTaskStatus(ref updateTaskStatus, pendingTest,
                ref validationErrors);

            Assert.IsTrue(validationErrors.Count == 0);
        }

        [Test]
        public void AddPendingSpecimenRowTest()
        {
            // TODO: Implement unit test for AddPendingSpecimenRowTest
        }

        [Test]
        public void AddPendingUnitRowTest()
        {
            // TODO: Implement unit test for AddPendingUnitRowTest
        }

        [Test]
        public void PendingSpecimenTestExistsTest()
        {
            // TODO: Implement unit test for PendingSpecimenTestExistsTest
        }

        [Test]
        public void PendingBloodUnitTestExistsTest()
        {
            // TODO: Implement unit test for PendingBloodUnitTestExistsTest
        }

        [Test]
        public void ProcessTestResultsTest_MultipleTestsExist()
        {
            BOL.OrderedTest tmpOT = DataCreator.CreateNewPatientDiagnosticOrder(true, Guid.NewGuid(), OrderableTest.ABS);
            DataCreator.CreateReflexTest(tmpOT, OrderableTest.RepeatABS);
            DataCreator.CreateReflexTest(tmpOT, OrderableTest.RepeatABS);

            BOL.Specimen tmpPS = new BOL.Specimen(tmpOT.Specimen.SpecimenGuid);

            string msg = DataCreator.GetHL7ORUMessage_ABS(tmpPS.SpecimenUid);
            HL7AIOruMessage hl7msg = new HL7AIOruMessage(msg);
            List<string> errs = PendingResultsController.ProcessTestResults(hl7msg, BloodTestPanelName.ABS);

            Assert.AreEqual(1, errs.Count, "Error Counts");
            Assert.IsTrue(errs[0].Contains(string.Format("Multiple Tests Exist for Specimen UID: {0} ", tmpPS.SpecimenUid)), "Message");
        }

        [Test]
        public void ProcessTestResultsTest_XM_Pass()
        {
            BOL.BloodUnit tmpBU = DataCreator.CreateRBCBloodUnit(false, ABO.A, RH.Positive);
            Guid psGuid = DataCreator.CreatePatientAssignment(tmpBU.BloodUnitGuid, Guid.NewGuid(), false);
            BOL.Specimen tmpPS = new BOL.Specimen(psGuid);

            string msg = DataCreator.GetHL7ORUMessage_XM(tmpPS.SpecimenUid, tmpBU.LabelUnitId);
            HL7AIOruMessage hl7msg = new HL7AIOruMessage(msg);
            List<string> errs = PendingResultsController.ProcessTestResults(hl7msg, BloodTestPanelName.XM);

            Assert.AreEqual(0, errs.Count, "XM Pass");
        }

        [Test]
        public void ProcessTestResultsTest_XM_PendingXM()
        {
            BOL.BloodUnit tmpBU = DataCreator.CreateRBCBloodUnit(false, ABO.A, RH.Positive);
            Guid psGuid = DataCreator.CreatePatientAssignment(tmpBU.BloodUnitGuid, Guid.NewGuid(), false);
            BOL.Specimen tmpPS = new BOL.Specimen(psGuid);
            BOL.OrderedTest tmpOT = DataCreator.CreateNewPatientDiagnosticOrder(true, tmpPS.Patient.PatientGuid, OrderableTest.TAS);
            DataCreator.CreateReflexTest(tmpOT, OrderableTest.RepeatXM);

            string msg = DataCreator.GetHL7ORUMessage_XM(tmpPS.SpecimenUid, tmpBU.LabelUnitId);
            HL7AIOruMessage hl7msg = new HL7AIOruMessage(msg);
            List<string> errs = PendingResultsController.ProcessTestResults(hl7msg, BloodTestPanelName.XM);

            Assert.AreEqual(0, errs.Count, "XM Pass");

            List<string> errs2 = PendingResultsController.ProcessTestResults(hl7msg, BloodTestPanelName.XM);

            Assert.AreEqual(1, errs2.Count, "XM Errors");
            Assert.AreEqual(string.Format("There is already a pending crossmatch test for Blood Unit ID: {0}.", tmpBU.LabelUnitId), errs2[0], "Message");
        }

        [Test]
        public void ProcessTestResultsTest_PendingBUT()
        {
            BOL.BloodUnit tmpBU = DataCreator.CreateRBCBloodUnit(false, ABO.A, RH.Positive);
            string msg = DataCreator.GetHL7ORUMessage_UnitABORh(tmpBU.LabelUnitId);
            HL7AIOruMessage hl7msg = new HL7AIOruMessage(msg);
            List<string> errs = PendingResultsController.ProcessTestResults(hl7msg, BloodTestPanelName.UnitABORh);

            Assert.AreEqual(0, errs.Count, "Unit ABORH Pass");

            List<string> errs2 = PendingResultsController.ProcessTestResults(hl7msg, BloodTestPanelName.UnitABORh);

            Assert.AreEqual(1, errs2.Count, "Errors");
            Assert.AreEqual(string.Format("There is already a pending blood unit test for Blood Unit ID: {0}.",tmpBU.LabelUnitId), errs2[0], "Message");
        }

        [Test]
        public void ProcessTestResultsTest_XM_Locked()
        {
            BOL.BloodUnit tmpBU = DataCreator.CreateRBCBloodUnit(false, ABO.A, RH.Positive);
            Guid psGuid = DataCreator.CreatePatientAssignment(tmpBU.BloodUnitGuid, Guid.NewGuid(), false);
            BOL.Specimen tmpPS = new BOL.Specimen(psGuid);

            DAL.LockManager.LockRecord(tmpBU.BloodUnitStatus.BloodUnitStatusGuid, -999, LockFunctions.UnitTests, false, -999, 15);

            string msg = DataCreator.GetHL7ORUMessage_XM(tmpPS.SpecimenUid, tmpBU.LabelUnitId);
            HL7AIOruMessage hl7msg = new HL7AIOruMessage(msg);
            List<string> errs = PendingResultsController.ProcessTestResults(hl7msg, BloodTestPanelName.XM);

            Assert.AreEqual(1, errs.Count, "XM Locked");
            Assert.AreEqual(StrRes.ValidMsg.Locking.PendingBloodUnitTestExists(tmpBU.LabelUnitId).ResString, errs[0], "Message");

            DAL.LockManager.UnlockAllRecordsForUser(Common.LogonUser.LogonUserName, true);
        }


        [Test]
        public void ProcessTestResultsTest_XM_BogusUnit()
        {
            Guid buGuid = Guid.NewGuid();
            Guid bpGuid = DataCreator.GetRandomBloodProduct(false, COMMON.ComponentClass.PLT).BloodProductGuid;  //platelets 
            BOL.BloodUnit tmpUnit = DataCreator.CreateBloodUnit(buGuid, Guid.NewGuid(), Guid.Empty, bpGuid, false);
            string msg = DataCreator.GetHL7ORUMessage_XM("000000001", tmpUnit.LabelUnitId);
            HL7AIOruMessage hl7msg = new HL7AIOruMessage(msg);
            List<string> errs = PendingResultsController.ProcessTestResults(hl7msg, BloodTestPanelName.XM);

            Assert.AreEqual(1, errs.Count, "Bogus Unit");
            Assert.AreEqual(string.Format("No Crossmatch Blood Unit  Available for Identifier: {0}.", tmpUnit.LabelUnitId), errs[0], "Message");
        }

        [Test]
        public void ProcessTestResultsTest_XM_BadSpecimen()
        {
            Hashtable guids = DataCreator.CreateCrossmatchedBloodUnitTestData();
            Guid buGuid = (Guid)guids["bloodUnitGuid"];
            BOL.BloodUnit tmpUnit = new BOL.BloodUnit(buGuid);

            string fakeUID = "98765420";
            string msg = DataCreator.GetHL7ORUMessage_XM(fakeUID, tmpUnit.LabelUnitId);
            HL7AIOruMessage hl7msg = new HL7AIOruMessage(msg);
            List<string> errs = PendingResultsController.ProcessTestResults(hl7msg, BloodTestPanelName.XM);

            Assert.AreEqual(1, errs.Count, "Invalid Specimen");//
            Assert.AreEqual(string.Format("No Crossmatch Specimen Available for Identifier: {0}.", fakeUID),errs[0], "Message");
        }

        [Test]
        public void ProcessTestResultsTest_TAS_Passed()
        {
            Guid pGuid = Guid.NewGuid();
            BOL.OrderedTest tmpOT = DataCreator.CreateNewPatientDiagnosticOrder(true, pGuid, OrderableTest.TAS);

            string msg = DataCreator.GetHL7ORUMessage_TAS(tmpOT.Specimen.SpecimenUid);
            HL7AIOruMessage hl7msg = new HL7AIOruMessage(msg);
            List<string> errs = PendingResultsController.ProcessTestResults(hl7msg, BloodTestPanelName.TAS);

            Assert.AreEqual(0, errs.Count, "Valid TAS processed");
        }

        [Test]
        public void ProcessTestResultsTest_TAS_Defect303325()
        {
            Guid pGuid = Guid.NewGuid();
            BOL.OrderedTest tmpOT = DataCreator.CreateNewPatientDiagnosticOrder(true, pGuid, OrderableTest.TAS);

            string msg = DataCreator.GetHL7ORUMessage_TAS(tmpOT.Specimen.SpecimenUid);
            HL7AIOruMessage hl7msg = new HL7AIOruMessage(msg+msg);
            List<string> errs = PendingResultsController.ProcessTestResults(hl7msg, BloodTestPanelName.TAS);

            Assert.AreNotEqual(0, errs.Count, "Errors Exist");
            Assert.AreEqual("Duplcate Tests in Message OBX.3.", errs[0], "Message");
        }

        [Test]
        public void ProcessTestResultsTest_TAS_NonFinalSpecimenActionCode()
        {
            Guid pGuid = Guid.NewGuid();
            BOL.OrderedTest tmpOT = DataCreator.CreateNewPatientDiagnosticOrder(true, pGuid, OrderableTest.TAS);

            string msg = DataCreator.GetHL7ORUMessage_TAS(tmpOT.Specimen.SpecimenUid);
            HL7AIOruMessage hl7msg = new HL7AIOruMessage(msg.Replace("|F|", "|O|"));
            List<string> errs = PendingResultsController.ProcessTestResults(hl7msg, BloodTestPanelName.TAS);

            Assert.AreNotEqual(0, errs.Count, "Errors Exist");
            Assert.AreEqual("Invalid Observation Result Status OBX.11.", errs[0], "Message");
        }


        [Test]
        public void ProcessTestResultsTest_PatientABORH_Pass()
        {
            Guid pGuid = Guid.NewGuid();
            BOL.OrderedTest tmpOT = DataCreator.CreateNewPatientDiagnosticOrder(true, pGuid, OrderableTest.ABORh);

            string msg = DataCreator.GetHL7ORUMessage_ABORH(tmpOT.Specimen.SpecimenUid);
            HL7AIOruMessage hl7msg = new HL7AIOruMessage(msg);
            List<string> errs = PendingResultsController.ProcessTestResults(hl7msg, BloodTestPanelName.PatientABORh);
            
            Assert.AreEqual(0, errs.Count, "Valid ABORH processed");
        }

        [Test]
        public void ProcessTestResultsTest_PatientABORH_CombinedInterpGoof()
        {
            Guid pGuid = Guid.NewGuid();
            BOL.OrderedTest tmpOT = DataCreator.CreateNewPatientDiagnosticOrder(true, pGuid, OrderableTest.ABORh);

            string msg = DataCreator.GetHL7ORUMessage_ABORH_CombinedInterp(tmpOT.Specimen.SpecimenUid).Replace("AB Neg", "Neg");
            HL7AIOruMessage hl7msg = new HL7AIOruMessage(msg);
            List<string> errs = PendingResultsController.ProcessTestResults(hl7msg, BloodTestPanelName.PatientABORh);

            Assert.AreEqual("Not supported ABORh intrepritaion invalid count:  on Test: Patient ABORh For Blood Test ABORhInterp.", errs[0], "Message");
        }

        [Test]
        public void ProcessTestResultsTest_UnitABORH_CombinedInterpGoof()
        {
            BOL.BloodUnit tmpBU = DataCreator.CreateRBCBloodUnit(false, ABO.A, RH.Positive);

            string msg = DataCreator.GetHL7ORUMessage_UnitABORhCombinedInterp(tmpBU.LabelUnitId).Replace("A Pos", "Oink");
            HL7AIOruMessage hl7msg = new HL7AIOruMessage(msg);
            List<string> errs = PendingResultsController.ProcessTestResults(hl7msg, BloodTestPanelName.UnitABORh);

            Assert.AreEqual("Not supported ABORh intrepritaion invalid count:  on Test: Unit ABORh For Blood Test ABORhInterp.", errs[0], "Message");
        }

        [Test]
        public void ProcessTestResultsTest_BadUser()
        {
            Guid pGuid = Guid.NewGuid();
            BOL.OrderedTest tmpOT = DataCreator.CreateNewPatientDiagnosticOrder(true, pGuid, OrderableTest.ABORh);

            string msg = DataCreator.GetHL7ORUMessage_ABORH_CombinedInterp(tmpOT.Specimen.SpecimenUid).Replace("DNS   vbecs1", "DNS   xxxxx");
            HL7AIOruMessage hl7msg = new HL7AIOruMessage(msg);
            List<string> errs = PendingResultsController.ProcessTestResults(hl7msg, BloodTestPanelName.PatientABORh);

            Assert.AreEqual("Invalid User ID: DNS   xxxxx on Test: AntiA.", errs[0], "Message");
        }

        [Test]
        public void ProcessTestResultsTest_DAT_Pass()
        {
            Guid pGuid = Guid.NewGuid();
            BOL.OrderedTest tmpOT = DataCreator.CreateNewPatientDiagnosticOrder(true, pGuid, OrderableTest.DAT);

            string msg = DataCreator.GetHL7ORUMessage_DAT(tmpOT.Specimen.SpecimenUid);
            HL7AIOruMessage hl7msg = new HL7AIOruMessage(msg);
            List<string> errs = PendingResultsController.ProcessTestResults(hl7msg, BloodTestPanelName.DAT);

            Assert.AreEqual(0, errs.Count, "Valid DAT processed");
        }

        [Test]
        public void ProcessTestResultsTest_UnitAntigenTyping_Pass()
        {
            BOL.BloodUnit tmpUnit = DataCreator.CreateRBCBloodUnit(false, ABO.A, RH.Positive);

            string msg = DataCreator.GetHL7ORUMessage_AntigenTyping(tmpUnit.LabelUnitId);
            HL7AIOruMessage hl7msg = new HL7AIOruMessage(msg);
            List<string> errs = PendingResultsController.ProcessTestResults(hl7msg, BloodTestPanelName.AnigenTyping);

            Assert.AreEqual(0, errs.Count, "Valid Unit AntigenTyping processed");
        }

        [Test]
        public void ProcessTestResultsTest_UnitAntigenTyping_BadMapping()
        {
            BOL.BloodUnit tmpUnit = DataCreator.CreateRBCBloodUnit(false, ABO.A, RH.Positive);

            string msg = DataCreator.GetHL7ORUMessage_AntigenTyping(tmpUnit.LabelUnitId).Replace("AntiC", "Barf");
            HL7AIOruMessage hl7msg = new HL7AIOruMessage(msg);
            List<string> errs = PendingResultsController.ProcessTestResults(hl7msg, BloodTestPanelName.AnigenTyping);

            Assert.AreEqual("PendingResultController.GetTestTypeMapping: Failed to find TestTypeId for TestName: Antigen Typing and TestType Barf.", errs[0]);
        }

        [Test]
        public void ProcessTestResultsTest_PatientAntigenTyping_Pass()
        {
            Guid pGuid = Guid.NewGuid();
            BOL.OrderedTest tmpOT = DataCreator.CreateNewPatientDiagnosticOrder(true, pGuid, OrderableTest.ABS);
            DataCreator.CreateReflexTest(tmpOT, OrderableTest.AGc, OrderableTest.AGC, OrderableTest.AGe, OrderableTest.AGE);
            string msg = DataCreator.GetHL7ORUMessage_AntigenTyping(tmpOT.Specimen.SpecimenUid);
            HL7AIOruMessage hl7msg = new HL7AIOruMessage(msg);
            List<string> errs = PendingResultsController.ProcessTestResults(hl7msg, BloodTestPanelName.AnigenTyping);

            Assert.AreEqual(0, errs.Count, "Valid Patient AntigenTyping processed");
        }

        [Test]
        public void ProcessTestResultsTest_TAS_MissingUser()
        {
            Guid pGuid = Guid.NewGuid();
            BOL.OrderedTest tmpOT = DataCreator.CreateNewPatientDiagnosticOrder(true, pGuid, OrderableTest.TAS);

            string msg = DataCreator.GetHL7ORUMessage_TAS(tmpOT.Specimen.SpecimenUid).Replace("DNS   vbecs1", "seemybutts");
            HL7AIOruMessage hl7msg = new HL7AIOruMessage(msg);
            List<string> errs = PendingResultsController.ProcessTestResults(hl7msg, BloodTestPanelName.TAS);

            Assert.IsTrue(errs.Count > 0, "Missing User");
            Assert.IsTrue(errs[0].Contains("Invalid User ID: seemybutts on Test"), "Message");
        }

        [Test]
        public void ProcessTestResultsTest_TAS_Missing_TestNames()
        {
            Guid pGuid = Guid.NewGuid();
            BOL.OrderedTest tmpOT = DataCreator.CreateNewPatientDiagnosticOrder(true, pGuid, OrderableTest.TAS);

            string msg = DataCreator.GetHL7ORUMessage_TAS(tmpOT.Specimen.SpecimenUid).Replace("Pos", "Poo");
            HL7AIOruMessage hl7msg = new HL7AIOruMessage(msg);
            List<string> errs = PendingResultsController.ProcessTestResults(hl7msg, BloodTestPanelName.TAS);

            Assert.IsTrue(errs.Count > 0, "Missing TestNames");
            Assert.IsTrue(errs[0].Contains("Not supported test interpritation ID: Poo"), "Message");
        }

        [Test]
        public void ProcessTestResultsTest_TAS_Missing_Results()
        {
            Guid pGuid = Guid.NewGuid();
            BOL.OrderedTest tmpOT = DataCreator.CreateNewPatientDiagnosticOrder(true, pGuid, OrderableTest.TAS);

            string msg = DataCreator.GetHL7ORUMessage_TAS(tmpOT.Specimen.SpecimenUid).Replace("4+", "Poo");
            HL7AIOruMessage hl7msg = new HL7AIOruMessage(msg);
            List<string> errs = PendingResultsController.ProcessTestResults(hl7msg, BloodTestPanelName.TAS);

            Assert.IsTrue(errs.Count > 0, "Missing Results");
            Assert.IsTrue(errs[0].Contains("Not supported"), "Message");
        }



        [Test]
        public void ProcessTestResultsTest_TAS_Locked()
        {
            Guid pGuid = Guid.NewGuid();
            BOL.OrderedTest tmpOT = DataCreator.CreateNewPatientDiagnosticOrder(true, pGuid, OrderableTest.TAS);

            DAL.LockManager.LockRecord(tmpOT.OrderedTestGuid, -999, LockFunctions.UnitTests, false, -999, 15);

            string msg = DataCreator.GetHL7ORUMessage_TAS(tmpOT.Specimen.SpecimenUid);
            HL7AIOruMessage hl7msg = new HL7AIOruMessage(msg);
            List<string> errs = PendingResultsController.ProcessTestResults(hl7msg, BloodTestPanelName.TAS);

            Assert.AreEqual(1, errs.Count, "TAS Locked");
            Assert.AreEqual(StrRes.ValidMsg.Locking.PendingSpecimenTestExists(tmpOT.Specimen.SpecimenUid).ResString, errs[0], "Message");

            DAL.LockManager.UnlockAllRecordsForUser(Common.LogonUser.LogonUserName, true);
        }

        [Test]
        public void ProcessTestResultsTest_SpecimenEqualsUnitID()
        {
            string fakeID = "FakeID";
            string msg = DataCreator.GetHL7ORUMessage_UnitABORh(fakeID + "^" + fakeID );
            HL7AIOruMessage hl7msg = new HL7AIOruMessage(msg);
            List<string> errs = PendingResultsController.ProcessTestResults(hl7msg, BloodTestPanelName.UnitABORh);

            Assert.AreEqual(1, errs.Count, "Unit=Specimen-impossible!");//
            Assert.AreEqual(string.Format("No Blood Unit Or Specimen Available for Identifier: {0}.", fakeID), errs[0], "Message");
        }

        [Test]
        public void ProcessTestResultsTest_NotSpecimen_NotUnit()
        {
            string fakeID = "Flight";
            string fakeUID = "815";
           
            string msg = DataCreator.GetHL7ORUMessage_UnitABORh(fakeUID + "^" + fakeID);
            HL7AIOruMessage hl7msg = new HL7AIOruMessage(msg);
            List<string> errs = PendingResultsController.ProcessTestResults(hl7msg, BloodTestPanelName.UnitABORh);

            Assert.AreEqual(1, errs.Count, "Unit=Specimen-impossible!");//
            Assert.AreEqual(string.Format("No Blood Unit Or Specimen Available for Specimen UID: {0} and Blood Unit ID: {1}.", fakeUID, fakeID), errs[0], "Message");
        }

        [Test]
        public void ProcessTestResultsTest_SpecimenTestExists()
        {
            Guid pGuid = Guid.NewGuid();
            BOL.OrderedTest tmpOT = DataCreator.CreateNewPatientDiagnosticOrder(true, pGuid, OrderableTest.ABORh);

            string msg = DataCreator.GetHL7ORUMessage_ABORH(tmpOT.Specimen.SpecimenUid);
            HL7AIOruMessage hl7msg = new HL7AIOruMessage(msg);
            List<string> errs = PendingResultsController.ProcessTestResults(hl7msg, BloodTestPanelName.PatientABORh);
            errs = PendingResultsController.ProcessTestResults(hl7msg, BloodTestPanelName.PatientABORh);
            
            Assert.AreEqual(1, errs.Count, "Error Count");
            
            string expected = string.Format("There is already a pending specimen test for Specimen UID: {0}", tmpOT.Specimen.SpecimenUid);
            Assert.IsTrue(errs[0].Contains(expected), "Message");
        }

        [Test]
        public void ProcessPendingTestResults_AntigenProcessing_ValidResults()
        {
            PendingResultsController.PendingBloodUnitTest pendingTest = new PendingResultsController.PendingBloodUnitTest();
            Guid bloodUnitGuid = Guid.NewGuid();
            pendingTest.OrderableTestName = "Antigen Typing";
            pendingTest.BloodUnitGuid = bloodUnitGuid;
            pendingTest.BloodTestTypeName = "AntiC";
            pendingTest.TestTechId = "ishdobrak";
            pendingTest.TestDate = HL7DateFormat.ConvertHL7DateTime("20160201160003");
            pendingTest.TestResultId = pendingTest.TestResultId;
            pendingTest.TestResultText = "3";
            pendingTest.AutoInstrumentName = "ProVueSanDiego";
            pendingTest.DivisionCode = "589";
            pendingTest.ResultSetGuid = Guid.NewGuid();

            int bloodTestTypeId = 0;
            int mappedTestTypeId = 0;
            List<string> validationErrors = new List<string>();
            bool success = PendingResultsController.GetTestTypeMapping(pendingTest.OrderableTestName, pendingTest.BloodTestTypeName, 0,
                out bloodTestTypeId, out mappedTestTypeId, ref validationErrors);

            Assert.IsTrue(success);
            Assert.IsTrue(validationErrors.Count == 0);
            Assert.IsTrue(bloodTestTypeId > 0);
            Assert.IsTrue(mappedTestTypeId > 0);
        }


	}
}
#endif
